# Really Simple JSON Web Tokens
[![Actions Status](https://github.com/robdwaller/reallysimplejwt/workflows/ci/badge.svg)](https://github.com/robdwaller/reallysimplejwt/actions) [![codecov](https://codecov.io/gh/RobDWaller/ReallySimpleJWT/branch/master/graph/badge.svg)](https://codecov.io/gh/RobDWaller/ReallySimpleJWT) [![Infection MSI](https://badge.stryker-mutator.io/github.com/RobDWaller/ReallySimpleJWT/master)](https://infection.github.io) [![StyleCI](https://styleci.io/repos/82379868/shield?branch=master)](https://styleci.io/repos/82379868) [![Latest Stable Version](https://poser.pugx.org/rbdwllr/reallysimplejwt/v/stable)](https://packagist.org/packages/rbdwllr/reallysimplejwt) ![Packagist PHP Version Support](https://img.shields.io/packagist/php-v/rbdwllr/reallysimplejwt) [![Total Downloads](https://poser.pugx.org/rbdwllr/reallysimplejwt/downloads)](https://packagist.org/packages/rbdwllr/reallysimplejwt)

A simple PHP library for creating JSON Web Tokens that uses HMAC SHA256 to sign signatures. For basic usage the library exposes a static interface to allow developers to create a token that stores a user identifier and expiration time.

The library is also open to extension, developers can define their own encoding and secret standards, set all the [RFC standard](https://tools.ietf.org/html/rfc7519) JWT claims, and set their own private claims.

You can easily integrate ReallySimpleJWT with PSR-7 / PSR-15 compliant frameworks such as [Slim PHP](https://packagist.org/packages/slim/slim) with the [PSR-JWT middleware library](https://github.com/RobDWaller/psr-jwt). Please read the [framework integration documentation](#framework-integration-with-psr-jwt-middleware) to learn more.

If you need to read tokens in the browser please take a look at our JavaScript / Typescript library [RS-JWT](https://github.com/RobDWaller/rs-jwt).

## Contents

- [What is a JSON Web Token?](#what-is-a-json-web-token)
- [Setup](#setup)
- [Basic Usage](#basic-usage)
    - [Create Token](#create-token)
    - [Validate Token](#validate-token)
    - [Get Header and Payload Claims Data](#get-header-and-payload-claims-data)
    - [Build, Parse and Validate Factory Methods](#build-parse-and-validate-factory-methods)
    - [Non-Static Usage](#non-static-usage)
- [Advanced Usage](#advanced-usage)
    - [Create Custom Token](#create-custom-token)
    - [Access the Token](#access-the-token)
    - [Parse Token](#parse-token)
    - [Access Token Claims Data](#access-token-claims-data)
    - [Token Validation Methods](#token-validation-methods)
    - [Custom Encoding](#custom-encoding)
- [Error Messages and Codes](#error-messages-and-codes)
- [Token Security](#token-security)
    - [Secret Strength](#secret-strength)
- [Framework Integration With PSR-JWT Middleware](#framework-integration-with-psr-jwt-middleware)
- [Browser Integration With RS-JWT](#browser-integration-with-rs-jwt)

## What is a JSON Web Token?

JSON Web Tokens is a standard for creating URL friendly access tokens that assert claims about a user or system.

A token is broken down into three parts; the header, the payload and the signature; with each part separated by a dot. Each part is encoded using the base64URL standard, see the [RFC](https://tools.ietf.org/html/rfc4648#page-7).

An example JWT:

```
eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJzdWIiOiIxMjM0NTY3ODkwIiwibmFtZSI6IkpvaG4gRG9lIiwiaWF0IjoxNTE2MjM5MDIyfQ.SflKxwRJSMeKKF2QT4fwpMeJf36POk6yJV_adQssw5c
```

The header and payload are both encoded JSON strings that contain a number of claims:

```javascript
// Example Header
{
  "alg": "HS256",
  "typ": "JWT"
}

// Example Payload
{
  "sub": "1234567890",
  "name": "John Doe",
  "iat": 1516239022
}
```

A claim is a key value pair, eg `"typ": "JWT"`, please read [RFC 7519](https://tools.ietf.org/html/rfc7519#section-4) to learn more about JSON Web Token claims.

Token security is achieved via the signature which is made up of the header, payload and a secret known only to the token author. This information is hashed and then base64URL encoded.

If a malicious user attempts to edit the header or payload claims they will be unable to replicate the signature so long as you use a strong secret. See [Token Security](#token-security) for more information on this.

## Setup

To install this package you will need to install [Composer](https://getcomposer.org/) and then run `composer init`. Once this is done you can install the package via the command line or by editing the composer.json file created by the `composer init` command.

Finally you will need to reference the composer autoloader in your PHP code, `require 'vendor/autoload.php';`. The location of the autoload file will differ dependent on where your code is run and may not be required if you are using a framework.

**Install via Composer on the command line:**

```bash
composer require rbdwllr/reallysimplejwt
```

**Install via the composer.json file:**

Add the following to your composer.json file:

```javascript
"require": {
    "rbdwllr/reallysimplejwt": "^5.0"
}
```

Then run:

```bash
composer update
```

## Basic Usage

For basic usage the library exposes a set of static methods via the `ReallySimpleJWT\Token` class which allow a developer to create and validate basic JSON Web Tokens.

### Create Token

Call the `create()` static method and pass in a user identifier, a secret, an expiration date time number and the token issuer.

This will return a token string on success and throw a `ReallySimpleJWT\Exception\BuildException` on failure.

```php
use ReallySimpleJWT\Token;

$userId = 12;
$secret = 'sec!ReT423*&';
$expiration = time() + 3600;
$issuer = 'localhost';

$token = Token::create($userId, $secret, $expiration, $issuer);
```

To create a more customised token developers can use the `customPayload()` method. This allows the creation of a token based on an array of key value pairs which represent the payload claims.

```php
use ReallySimpleJWT\Token;

$payload = [
    'iat' => time(),
    'uid' => 1,
    'exp' => time() + 10,
    'iss' => 'localhost'
];

$secret = 'Hello&MikeFooBar123';

$token = Token::customPayload($payload, $secret);
```

On success the `customPayload()` method will return a JWT token string and on failure it will throw an exception.

### Validate Token

To validate a JSON web token call the `validate()` static method, pass in the token string and the secret. The validate method checks the token structure is correct and the signature is valid.

It will return true on success and false on failure.

```php
use ReallySimpleJWT\Token;

$token = 'aaa.bbb.ccc';
$secret = 'sec!ReT423*&';

$result = Token::validate($token, $secret);
```

There are also methods available to validate the token's expiration claim and not before claim. Both will return true on success and false on failure.

```php
use ReallySimpleJWT\Token;

$token = 'aaa.bbb.ccc';

Token::validateExpiration($token);

Token::validateNotBefore($token);
```

### Get Header and Payload Claims Data

To retrieve the token claims data from the header or payload call the `getHeader()` and or `getPayload()` static methods.

Both methods will return an associative array on success and throw an exception on failure.

```php
use ReallySimpleJWT\Token;

$token = 'aaa.bbb.ccc';

// Return the header claims
Token::getHeader($token);

// Return the payload claims
Token::getPayload($token);
```

### Build, Parse and Validate Factory Methods

The `ReallySimpleJWT\Token` class also provides three factory methods to gain access to the core `ReallySimpleJWT\Build`, `ReallySimpleJWT\Parse`, and `ReallySimpleJWT\Validate` classes. These classes allow you to build custom tokens, and parse and validate tokens as you see fit.

```php
Token::builder(); // Returns an instance of ReallySimpleJWT\Build

Token::parser($token); // Returns an instance of ReallySimpleJWT\Parse

Token::validator($token, $secret); // Returns an instance of ReallySimpleJWT\Validate
```

### Non-Static Usage

The `ReallySimpleJWT\Token` class is just a wrapper for the `ReallySimpleJWT\Tokens` class which can be used directly for those who'd prefer to instantiate and inject the functionality.

```php
use ReallySimpleJWT\Tokens;

$tokens = new Tokens();

$id = 52;
$secret = 'sec!ReT423*&';
$expiration = time() + 50;
$issuer = 'localhost';

$token = $tokens->create('id', $id, $secret, $expiration, $issuer);
$token->getToken();
```

Please note when calling the `create()` and `customPayload()` methods on the `Tokens` class they will return an instance of the `Jwt` class unlike the `Token` class which will return a token string.

In addition, the `create()` method has a slightly different signature to the `Tokens` class as a user identifier key must be passed in.

`create(string $userKey, $userId, string $secret, int $expiration, string $issuer): Jwt`

## Advanced Usage

To create customised JSON Web Tokens developers need to access the `ReallySimpleJWT\Build`, `ReallySimpleJWT\Parse` and `ReallySimpleJWT\Validate` classes directly.

### Create Custom Token

The `ReallySimpleJWT\Build` class allows you to create a completely unique JSON Web Token. It has helper methods for all the [RFC](https://tools.ietf.org/html/rfc7519#section-4) defined header and payload claims. For example, the `setIssuer()` method will add the `iss` claim to the token payload.

The class also allows developers to set custom header and payload claims via the `setHeaderClaim()` and `setPayloadClaim()` methods.

The methods can be chained together and when the `build()` method is called the token will be generated and returned as a `ReallySimpleJWT\Jwt` object.

```php
use ReallySimpleJWT\Build;
use ReallySimpleJWT\Helper\Validator;
use ReallySimpleJWT\Encoders\EncodeHS256;

$secret = '!secReT$123*';

$build = new Build('JWT', new Validator(), new EncodeHS256($secret));

$token = $build->setContentType('JWT')
    ->setHeaderClaim('info', 'foo')
    ->setIssuer('localhost')
    ->setSubject('admins')
    ->setAudience('https://google.com')
    ->setExpiration(time() + 30)
    ->setNotBefore(time() - 30)
    ->setIssuedAt(time())
    ->setJwtId('123ABC')
    ->setPayloadClaim('uid', 12)
    ->build();
```

### Access the Token

A `ReallySimpleJWT\Jwt` object is returned when a developer calls the `build()` method on the `ReallySimpleJWT\Build` class. The Jwt class offers a single `getToken()` method which returns the token string.

To parse a JSON Web Token via the `ReallySimpleJWT\Parse` class a developer must first create a new `ReallySimpleJWT\Jwt` object by injecting the token string on instantiation. The Jwt class will validate the structure of the token on instantiation to ensure integrity.

```php
use ReallySimpleJWT\Jwt;

$token = 'aaa.bbb.ccc';

$jwt = new Jwt($token);

// Return the token
$jwt->getToken();
```

### Parse Token

The `ReallySimpleJWT\Parse` class allows a developer to parse a JWT and the `parse()` method will decode the JSON Web Token and return the result as a `ReallySimpleJWT\Parsed` object. This will provide access to the header and payload claims data the token holds.

```php
use ReallySimpleJWT\Parse;
use ReallySimpleJWT\Jwt;
use ReallySimpleJWT\Decode;

$token = 'aaa.bbb.ccc';

$jwt = new Jwt($token);

$parse = new Parse($jwt, new Decode());

$parsed = $parse->parse();

// Return the token header claims as an associative array.
$parsed->getHeader();

// Return the token payload claims as an associative array.
$parsed->getPayload();
```

### Access Token Claims Data

The `ReallySimpleJWT\Parsed` class is returned when a developer calls the `parse()` method on the `ReallySimpleJWT\Parse` class.

It provides a number of helper methods to gain access to the token claim data. A developer can call the `getHeader()` and `getPayload()` methods to gain access to the respective claim data as associative arrays.

Alternatively a developer can call one of the [RFC](https://tools.ietf.org/html/rfc7519#section-4) compliant claim methods:

**Header**
- `getAlgorithm()`
- `getType()`
- `getContentType()`

**Payload**
- `getIssuer()`
- `getSubject()`
- `getAudience()`
- `getExpiration()`
- `getNotBefore()`
- `getIssuedAt()`
- `getJwtId()`
- `getExpiresIn()`
- `getUsableIn()`

### Token Validation Methods

To Validate a JSON Web Token a developer can use the `ReallySimpleJWT\Validate` class. To use the validate class you need to create and inject a `ReallySimpleJWT\Parsed` object. This is so the validate class can access the information contained within the token. 

```php
use ReallySimpleJWT\Jwt;
use ReallySimpleJWT\Parse;
use ReallySimpleJWT\Validate;
use ReallySimpleJwt\Decode;
use ReallySimpleJwt\Encoders\EncodeHS256;
use ReallySimpleJwt\Helper\Validator;

$token = new Jwt('abc.def.ghi');

$parse = new Parse($jwt, new Decode());

$parsed = $parse->parse();

$validate = new Validate(
    $parsed,
    new EncodeHS256(),
    new Validator()
);

$validate->signature();
```

Six validation methods are available which can all be chained:

- `signature()` confirms the token signature is valid.
- `expiration()` confirms the token expiration claim (`exp`) has not expired.
- `notBefore()` confirms the token not before claim (`nbf`) has elapsed.
- `audience()` confirms the token audience claim (`aud`) matches what is expected.
- `algorithm()` confirms the token algorithm claim (`alg`) matches what is expected and is valid (See: [RFC 7518](https://www.rfc-editor.org/rfc/rfc7518.html)).
- `algorithmNotNone()` confirms the token algorithm claim (`alg`) is not set to none.

Each validation method will throw a `ReallySimpleJWT\Exception\ValidateException` if there is anything wrong with the supplied token.

### Custom Encoding

By default this library hashes and encodes the JWT signature via `hash_hmac()` using the sha256 algorithm. If a developer would like to use a customised form of encoding they just need to generate a custom encode class which complies with the `ReallySimpleJWT\Interfaces\Encode` interface. This can then be injected into the `ReallySimpleJWT\Build` and `ReallySimpleJWT\Validate` classes.

```php
interface EncodeInterface
{
    public function getAlgorithm(): string;

    public function encode(string $toEncode): string;

    public function signature(string $header, string $payload): string;
}
```

## Error Messages and Codes

The ReallySimpleJWT library will in a number of situations throw exceptions to highlight problems when creating, parsing and validating JWT tokens. The error codes, messages and their explanations are in the table below.

There are six exception types that may be thrown:
- `ReallySimpleJWT\Exception\BuildException`
- `ReallySimpleJWT\Exception\EncodeException`
- `ReallySimpleJWT\Exception\JwtException`
- `ReallySimpleJWT\Exception\ParsedException`
- `ReallySimpleJWT\Exception\TokensException`
- `ReallySimpleJWT\Exception\ValidateException`

| Code | Message                           | Explanation                                |
|:----:| --------------------------------- | ------------------------------------------ |
| 1    | Token has an invalid structure.   | Token must have three parts separated by dots. |
| 2    | Audience claim does not contain provided StringOrURI.        | The `aud` claim must contain the provided string or URI string provided. |
| 3    | Signature is invalid.             | Signature does not match header / payload content. Could not replicate signature with provided header, payload and secret. |
| 4    | Expiration claim has expired.     | The `exp` claim must be a valid date time number in the future. |
| 5    | Not Before claim has not elapsed. | The `nbf` claim must be a valid date time number in the past. |
| 6    | The header claim [~claim~] is not set. | Attempt was made to access a header claim which does not exist. |
| 7    | The payload claim [~claim~] is not set. | Attempt was made to access a payload claim which does not exist. |
| 8    | Invalid payload claim.            | Payload claims must be key value pairs of the format `string: mixed`. |
| 9    | Invalid secret.                   | Must be 12 characters in length, contain upper and lower case letters, a number, and a special character `*&!@%^#$` |
| 10   | Algorithm claim is not valid.   | Algorithm should be a valid Digital Signature or MAC Algorithm, or none. See [RFC 7518](https://tools.ietf.org/html/rfc7518). |
| 11   | Algorithm claim should not be none.      | The `alg` claim should not be set to none. |

## Token Security

The JWT [RFC 7519](https://tools.ietf.org/html/rfc7519#section-7) allows for the creation of tokens without signatures and without secured / hashed signatures. The ReallySimpleJWT library however imposes security by default as there is no logical reason not to. All created tokens must have a signature and a strong secret, but the library will parse and validate tokens without a secret or a strong secret. The library will not validate tokens without a signature.

By default The ReallySimpleJWT library makes available two encoding implementations, `ReallySimpleJWT\Encoders\EncodeHS256` and `ReallySimpleJWT\Encoders\EncodeHS256Strong`. The latter enforces strict secret security and is used by default to create tokens via the `Token` and `Tokens` classes. The `EncodeHS256` does not impose strict secret security and can be used with the `Build` class to create tokens when required. In addition,
it is possible to create a custom encode class by implementing the `ReallySimpleJWT\Interfaces\Encode` interface. See the section [Custom Encoding](#custom-encoding).


### Secret Strength

This JWT library imposes strict secret security via the `EncodeHS256Strong` class. The secret provided must be at least 12 characters in length; contain numbers; upper and lowercase letters; and one of the following special characters `*&!@%^#$`.

```php
// Bad Secret
secret123

// Good Secret
sec!ReT423*&
```

The reason for this is that there are lots of [JWT Crackers](https://github.com/lmammino/jwt-cracker) available meaning weak secrets are easy to crack thus rendering the signature security JWT offers useless.

## Framework Integration With PSR-JWT Middleware

You can easily integrate ReallySimpleJWT with [PSR-7 / PSR-15](https://www.php-fig.org/psr/psr-15/) compliant frameworks such as [Slim PHP](https://packagist.org/packages/slim/slim) and Laminas by using the [PSR-JWT library](https://github.com/RobDWaller/psr-jwt).

For example integration with Slim PHP only requires a few lines of code:

```php
// Can be added to any routes file in Slim, often index.php.
require '../../vendor/autoload.php';

$app->get('/jwt', function (Request $request, Response $response) {
    $response->getBody()->write("JSON Web Token is Valid!");

    return $response;
})->add(\PsrJwt\Factory\JwtAuth::middleware('Secret123!456$', 'jwt', 'Authentication Failed'));
```

Please read the [PSR-JWT documentation](https://github.com/RobDWaller/psr-jwt) to learn more about integration options for ReallySimpleJWT.

## Browser Integration With RS-JWT

When you create JSON Web Tokens you may wish to read some of the information contained in the header and payload claims in the browser.

If you do, we have an NPM packages for that called [RS-JWT](https://github.com/RobDWaller/rs-jwt).

**Install:**
```bash
npm install --save rs-jwt
```

**Usage:**
```js
import { parseJwt } from 'rs-jwt'

const result = parseJwt('json.web.token')

// Return the header claims as an object.
const header = result.getHeader()

// Access the type claim.
console.log(header.typ)

// Return the payload claims as an object.
const payload = result.getPayload()

// Access the expiry claim.
console.log(payload.exp)
```

For more information see the project [README](https://github.com/RobDWaller/rs-jwt/blob/master/README.md) or visit the [NPM Page](https://www.npmjs.com/package/rs-jwt).

## License

MIT

## Author

Rob Waller

Twitter: [@RobDWaller](https://twitter.com/RobDWaller)
